--[[
 * rtp_h265_extractor.lua
 * wireshark plugin to extract h265/HEVC stream from RTP packets
 * 
 * Based on rtp_h264_extractor.lua by Volvet Zhang
 * Adapted for H.265/HEVC RTP payload format (RFC 7798)
 *
 * This plugin is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License.
 *]]

do
    local MAX_JITTER_SIZE = 50
    local h265_data = Field.new("h265")
    local rtp_seq = Field.new("rtp.seq")
	
    local function extract_h265_from_rtp()
        local function dump_filter(fd)
            local fh = "h265";
            if fd ~= nil and fd ~= "" then
                return string.format("%s and (%s)", fh, fd)
            else    
                return fh
            end
        end

        local h265_tap = Listener.new("ip", dump_filter(get_filter()))
        local text_window = TextWindow.new("H.265 extractor")
        local filename = ""
        local seq_payload_table = { }
        local pass = 0
        local packet_count = 0
        local max_packet_count = 0
        local fu_info = nil
        local pre_seq = 0
        
        -- 错误计数器，避免重复弹窗
        local error_counts = {
            incomplete_fu = 0,
            incomplete_fu_seq_gap = 0,
            payload_mismatch = 0,
            unsupported_nal = 0,
            invalid_ap_size = 0
        }
		
        local function log(info)
            text_window:append(info)
            text_window:append("\n")
        end
        
        -- get_preference is only available since 3.5.0
        if get_preference then
            local fileopen_dir = get_preference("gui.fileopen.dir")
            if fileopen_dir == '' then
                log("Wireshark preference 'gui.fileopen.dir' is not set, aborting.")
                return
            end
            filename = fileopen_dir  .. "/" .. os.date("video_%Y%m%d-%H%M%S.265")
        else
            filename = "dump.265"
        end
        
        log("Dumping H.265 stream to " .. filename)
        local fp = io.open(filename, "wb")
        if fp == nil then 
            log("Failed to open dump file '" .. filename .. "'")
            return
        end
        
        local function seq_compare(left, right)  
            if math.abs(right.key - left.key) < 1000 then  
                return left.key < right.key  
            else 
                return left.key > right.key  
            end  
        end  
        
        local function dump_single_nal(h265_payload)
            if fp == nil then
                return
            end
            fp:write("\00\00\00\01")
            fp:write(h265_payload:tvb()():raw())
            fp:flush()
        end
        
        local function dump_fu(fu_info) 
            if fu_info.complete == true then 
                if fp == nil then
                    return
                end
                log("dump_fu (H.265)")
                fp:write("\00\00\00\01")
                -- Write reconstructed NAL header (2 bytes for H.265)
                fp:write(string.char(fu_info.nal_header_byte1))
                fp:write(string.char(fu_info.nal_header_byte2))
            
                for i, obj in ipairs(fu_info.payloads) do
                    -- Skip FU header (3 bytes: PayloadHdr + FU header)
                    fp:write(obj:tvb()():raw(3))
                end
                fp:flush()
            else
                error_counts.incomplete_fu = error_counts.incomplete_fu + 1
                if error_counts.incomplete_fu == 1 then
                    log("Incomplete NAL from FUs, dropped (further similar errors will be counted silently)")
                end
            end
        end
        
        -- H.265 FU packet handling (RFC 7798)
        local function handle_fu(seq, h265_data)
            -- H.265 NAL unit header is 2 bytes
            local nal_byte1 = h265_data:get_index(0)
            local nal_byte2 = h265_data:get_index(1)
            local nal_type = bit.band(bit.rshift(nal_byte1, 1), 0x3F)
            
            -- FU type is 49
            if nal_type ~= 49 then
                return false
            end
            
            local fu_header = h265_data:get_index(2)
            local fu_type = bit.band(fu_header, 0x3F)
            local S_bit = bit.band(fu_header, 0x80) -- Start bit
            local E_bit = bit.band(fu_header, 0x40) -- End bit
            
            if S_bit ~= 0 then
                -- FU start
                fu_info = { }
                fu_info.payloads = { }
                fu_info.seq = seq
                fu_info.complete = true
                -- Reconstruct NAL header: keep F bit, LayerId, TID from FU header, replace Type with fu_type
                fu_info.nal_header_byte1 = bit.bor(bit.band(nal_byte1, 0x81), bit.lshift(fu_type, 1))
                fu_info.nal_header_byte2 = nal_byte2
                
                table.insert(fu_info.payloads, h265_data)
                log("FU start: seq = "..tostring(seq)..", type = "..tostring(fu_type))
                return true
            end
            
            if fu_info == nil then 
                error_counts.incomplete_fu = error_counts.incomplete_fu + 1
                if error_counts.incomplete_fu == 1 then
                    log("Incomplete FU found: No start flag, dropped (first occurrence)")
                end
                return true
            end
            
            if seq ~= (fu_info.seq + 1) % 65536 then
                error_counts.incomplete_fu_seq_gap = error_counts.incomplete_fu_seq_gap + 1
                if error_counts.incomplete_fu_seq_gap == 1 then
                    log("Incomplete FU found: sequence gap (first: expected "..tostring((fu_info.seq + 1) % 65536)..", got "..tostring(seq)..")")
                end
                fu_info.complete = false
                return true
            end
            
            fu_info.seq = seq
            table.insert(fu_info.payloads, h265_data)
            
            if E_bit ~= 0 then
                -- FU end
                log("FU stop: seq = "..tostring(seq))
                dump_fu(fu_info)
                fu_info = nil
            end 
            
            return true
        end
        
        -- H.265 AP (Aggregation Packet) handling (RFC 7798)
        local function handle_ap(h265_data)
            if fp == nil then
                return
            end
            log("Start dump AP NALs (H.265)")
            local offset = 2  -- Skip PayloadHdr (2 bytes for H.265)
            repeat
                if offset + 2 > h265_data:tvb():len() then
                    break
                end
                local size = h265_data:tvb()(offset, 2):uint()
                offset = offset + 2
                if offset + size > h265_data:tvb():len() then
                    error_counts.invalid_ap_size = error_counts.invalid_ap_size + 1
                    if error_counts.invalid_ap_size == 1 then
                        log("AP: invalid size detected, stopped (first occurrence)")
                    end
                    break
                end
                local next_nal_byte1 = h265_data:get_index(offset)
                local next_nal_type = bit.band(bit.rshift(next_nal_byte1, 1), 0x3F)
                log("AP has NAL type = "..next_nal_type..", size = "..size)
                fp:write("\00\00\00\01")
                fp:write(h265_data:tvb()():raw(offset, size))
                offset = offset + size
            until offset >= h265_data:tvb():len()
            fp:flush()
            log("Finish dump AP NALs")
        end
		
        local function on_ordered_h265_payload(seq, h265_data)
            -- H.265 NAL header is 2 bytes
            if h265_data:len() < 2 then
                return
            end
            local nal_byte1 = h265_data:get_index(0)
            local naltype = bit.band(bit.rshift(nal_byte1, 1), 0x3F)
            
            if naltype >= 0 and naltype <= 47 then 
                -- Single NAL unit packet
                if fu_info ~= nil then
                    log("Incomplete FU found, dropped")
                    fu_info = nil
                end
                dump_single_nal(h265_data)
            elseif naltype == 49 then
                -- FU (Fragmentation Unit)
                handle_fu(seq, h265_data)
            elseif naltype == 48 then
                -- AP (Aggregation Packet)
                if fu_info ~= nil then
                    error_counts.incomplete_fu = error_counts.incomplete_fu + 1
                    if error_counts.incomplete_fu == 1 then
                        log("Incomplete FU found when processing AP (first occurrence)")
                    end
                    fu_info = nil
                end
                handle_ap(h265_data)
            else
                error_counts.unsupported_nal = error_counts.unsupported_nal + 1
                if error_counts.unsupported_nal == 1 then
                    log("Unsupported NAL type = "..tostring(naltype).." (first occurrence)")
                end
            end 
        end
        
        local function on_jitter_buffer_output()
            table.sort(seq_payload_table, seq_compare)
            
            if #seq_payload_table > 0 then
                log("on_jitter_buffer_output: seq = "..tostring(seq_payload_table[1].key)..", payload len = "..tostring(seq_payload_table[1].value:len()))
                on_ordered_h265_payload(seq_payload_table[1].key, seq_payload_table[1].value)
                table.remove(seq_payload_table, 1)
            end
        end
        
        local function jitter_buffer_finalize() 
            for i, obj in ipairs(seq_payload_table) do
                log("jitter_buffer_finalize: seq = "..tostring(obj.key)..", payload len = "..tostring(obj.value:len()))
                on_ordered_h265_payload(obj.key, obj.value)
            end
        end
        
        local function on_h265_rtp_payload(seq, payload)
            local cur_seq = seq.value
            if packet_count == 0 then
                pre_seq = cur_seq
            else
                if cur_seq == pre_seq then
                    packet_count = packet_count + 1
                    return
                else
                    pre_seq = cur_seq
                end
            end

            packet_count = packet_count + 1
            table.insert(seq_payload_table, { key = tonumber(seq.value), value = payload.value })
            
            if #seq_payload_table > MAX_JITTER_SIZE then
                on_jitter_buffer_output()
            end
        end
        
        function h265_tap.packet(pinfo, tvb)
            local payloadTable = { h265_data() }
            local seqTable = { rtp_seq() }
            
            if (#payloadTable) < (#seqTable) then 
                error_counts.payload_mismatch = error_counts.payload_mismatch + 1
                if error_counts.payload_mismatch == 1 then
                    log("ERROR: payloadTable size is "..tostring(#payloadTable)..", seqTable size is "..tostring(#seqTable).." (first occurrence)")
                end
                return
            end
            
            if pass == 0 then 
                for i, payload in ipairs(payloadTable) do
                    max_packet_count = max_packet_count + 1
                end
            else 
                for i, payload in ipairs(payloadTable) do
                    on_h265_rtp_payload(seqTable[1], payload)
                end
                
                if packet_count == max_packet_count then
                    jitter_buffer_finalize()
                end
            end 
        end
		
        function h265_tap.reset()
        end
		
        function h265_tap.draw() 
        end
		
        local function remove() 
            if fp then 
                fp:close()
                fp = nil
            end
            h265_tap:remove()
        end 
		
        log("Start H.265 extraction")
        text_window:set_atclose(remove)
		
        log("Phase 1: Counting packets")
        pass = 0
        retap_packets()
        
        log("Phase 2: Extracting stream (max_packet_count = "..tostring(max_packet_count)..")")
        pass = 1
        retap_packets()

        if fp ~= nil then 
           fp:close()
           fp = nil
           log("H.265 video stream written to " .. filename)
        end
        
        -- 汇总错误统计
        log("\n=== Error Summary ===")
        if error_counts.incomplete_fu > 0 then
            log("Incomplete FU packets: " .. tostring(error_counts.incomplete_fu) .. " occurrences")
        end
        if error_counts.incomplete_fu_seq_gap > 0 then
            log("FU sequence gaps: " .. tostring(error_counts.incomplete_fu_seq_gap) .. " occurrences")
        end
        if error_counts.invalid_ap_size > 0 then
            log("Invalid AP sizes: " .. tostring(error_counts.invalid_ap_size) .. " occurrences")
        end
        if error_counts.unsupported_nal > 0 then
            log("Unsupported NAL types: " .. tostring(error_counts.unsupported_nal) .. " occurrences")
        end
        if error_counts.payload_mismatch > 0 then
            log("Payload/Seq table mismatches: " .. tostring(error_counts.payload_mismatch) .. " occurrences")
        end
        local total_errors = error_counts.incomplete_fu + error_counts.incomplete_fu_seq_gap + 
                             error_counts.invalid_ap_size + error_counts.unsupported_nal + 
                             error_counts.payload_mismatch
        if total_errors == 0 then
            log("No errors encountered during extraction")
        else
            log("Total errors: " .. tostring(total_errors))
        end
        
        log("Extraction complete")
	end

    register_menu("Extract H.265 stream from RTP", extract_h265_from_rtp, MENU_TOOLS_UNSORTED)
end
